<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Services\AiContextService;
use App\Services\AiSchemaService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use App\Models\AiMessage;
use Illuminate\Support\Facades\Cache;

class AiChatController extends Controller
{
    public function dashboard(Request $request)
    {
        $user = $request->user();
        if (!$user) abort(401);
        // Admin or Accounting only
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        // Last 100 AI chats
        $messages = AiMessage::query()
            ->with(['user:id,first_name,last_name,username'])
            ->orderByDesc('id')
            ->limit(100)
            ->get(['id','user_id','business_id','supplier_id','scope','intent','time_label','question','answer','ok','rating','created_at']);

        // FAQs
        $faqs = app(\App\Services\FaqService::class)->all();

        return view('ai.dashboard', compact('messages','faqs'));
    }

    public function saveFaq(Request $request)
    {
        $user = $request->user();
        if (!$user) abort(401);
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        $data = $request->validate([
            'id' => 'required|string|max:64',
            'patterns' => 'nullable|string', // comma or newline separated
            'answer' => 'required|string',
            'answer_translations.tl' => 'nullable|string',
            'answer_translations.ceb' => 'nullable|string',
        ]);
        $patterns = [];
        if (!empty($data['patterns'])) {
            $patterns = preg_split('/[\r\n,]+/', (string)$data['patterns']);
        }
        $ok = app(\App\Services\FaqService::class)->upsert([
            'id' => $data['id'],
            'patterns' => $patterns,
            'answer' => $data['answer'],
            'answer_translations' => (array) ($data['answer_translations'] ?? []),
        ]);
        if ($request->wantsJson()) return response()->json(['ok' => $ok]);
        return redirect()->route('ai.dashboard')->with('status', ['success' => 1, 'msg' => 'FAQ saved']);
    }

    public function getFaq(Request $request, string $id)
    {
        $user = $request->user();
        if (!$user) abort(401);
        // same permission check as dashboard
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        $faq = app(\App\Services\FaqService::class)->find($id);
        if (!$faq) return response()->json(['ok' => false], 404);
        $allowDelete = (($faq['source'] ?? '') === 'storage');
        return response()->json(['ok' => true, 'faq' => $faq, 'allow_delete' => $allowDelete]);
    }

    public function deleteFaq(Request $request)
    {
        $user = $request->user();
        if (!$user) abort(401);
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        $data = $request->validate(['id' => 'required|string|max:64']);
        $svc = app(\App\Services\FaqService::class);
        $ok = $svc->delete($data['id']);
        if ($request->wantsJson()) return response()->json(['ok' => $ok]);
        return redirect()->route('ai.dashboard')->with('status', ['success' => $ok?1:0, 'msg' => $ok ? 'FAQ deleted' : 'FAQ not found or not deletable']);
    }

    public function syncProducts(Request $request)
    {
        $user = $request->user();
        if (!$user) abort(401);
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        $businessId = (int) ($user->business_id ?? session('business.id'));
        if (!$businessId) abort(400, 'Missing business');

        // Lightweight server-side rate limit to avoid 429 from middleware and protect DB
        $lockKey = 'ai:sync_products:lock:' . $businessId;
        if (!Cache::add($lockKey, 1, now()->addSeconds(15))) {
            if ($request->wantsJson()) {
                return response()->json(['ok' => false, 'message' => 'Sync is already in progress. Please try again in a few seconds.'], 429);
            }
            return redirect()->route('ai.dashboard')->with('status', ['success' => 0, 'msg' => 'Sync is already in progress. Please try again in a few seconds.']);
        }
        try {

        // Build a light-weight product snapshot for AI consumption
        $rows = \DB::table('products as p')
            ->join('variations as v', 'p.id', '=', 'v.product_id')
            ->leftJoin('product_variations as pv', 'v.product_variation_id', '=', 'pv.id')
            ->leftJoin('variation_location_details as vld', 'v.id', '=', 'vld.variation_id')
            ->leftJoin('business_locations as bl', 'vld.location_id', '=', 'bl.id')
            ->where('p.business_id', $businessId)
            ->groupBy('v.id', 'p.id', 'bl.id')
            ->select(
                'p.id as product_id', 'p.name as product_name', 'p.sku as product_sku', 'p.type as product_type',
                'v.id as variation_id', 'v.name as variation_name', 'v.sub_sku', 'v.sell_price_inc_tax',
                'pv.name as product_variation_name',
                'bl.id as location_id', 'bl.name as location_name',
                \DB::raw('SUM(COALESCE(vld.qty_available,0)) as qty_available')
            )
            ->get();

        $snapshot = [];
        foreach ($rows as $r) {
            $vid = (int) $r->variation_id;
            if (!isset($snapshot[$vid])) {
                $snapshot[$vid] = [
                    'product' => trim($r->product_name),
                    'sku' => $r->sub_sku ?: ($r->product_sku ?: null),
                    'variation' => $r->variation_name ?: null,
                    'group' => $r->product_variation_name ?: null,
                    'price_inc_tax' => (float) $r->sell_price_inc_tax,
                    'group_prices' => [],
                    'locations' => [],
                ];
            }
            if ($r->location_id) {
                $snapshot[$vid]['locations'][] = [
                    'id' => (int) $r->location_id,
                    'name' => (string) $r->location_name,
                    'qty' => (float) $r->qty_available,
                ];
            }
        }

        // Attach group prices per variation
        if (!empty($snapshot)) {
            $varIds = array_map('intval', array_keys($snapshot));
            $gps = \DB::table('variation_group_prices as vgp')
                ->join('selling_price_groups as spg','spg.id','=','vgp.price_group_id')
                ->whereIn('vgp.variation_id', $varIds)
                ->select('vgp.variation_id','spg.name as group_name','vgp.price_inc_tax','vgp.price_type')
                ->get();
            $util = new \App\Utils\Util();
            foreach ($gps as $g) {
                $vid = (int) $g->variation_id;
                if (!isset($snapshot[$vid])) continue;
                $base = (float) $snapshot[$vid]['price_inc_tax'];
                $price = ($g->price_type === 'percentage') ? $util->calc_percentage($base, (float)$g->price_inc_tax) : (float)$g->price_inc_tax;
                $snapshot[$vid]['group_prices'][] = [
                    'name' => (string) $g->group_name,
                    'price_inc_tax' => $price,
                ];
            }
        }

        \Storage::disk('local')->put('ai/products_snapshot.json', json_encode(array_values($snapshot), JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
        return $request->wantsJson()
            ? response()->json(['ok' => true, 'count' => count($snapshot)])
            : redirect()->route('ai.dashboard')->with('status', ['success' => 1, 'msg' => 'Products synced: '.count($snapshot)]);
        } finally {
            Cache::forget($lockKey);
        }
    }

    public function trainReports(Request $request)
    {
        $user = $request->user();
        if (!$user) abort(401);
        $isAllowed = false;
        try {
            if (method_exists($user, 'getRoleNames')) {
                $roles = collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); });
                $biz = strtolower((string) ($user->business_id ?? ''));
                $isAllowed = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#'.$biz)
                    || $roles->contains('accounting') || $roles->contains('accounting#'.$biz);
            }
        } catch (\Throwable $e) {}
        abort_unless($isAllowed, 403);

        $businessId = (int) ($user->business_id ?? session('business.id'));
        if (!$businessId) abort(400, 'Missing business');

        $ranges = [
            'Today' => [now()->startOfDay(), now()->endOfDay()],
            'This Week' => [now()->startOfWeek(), now()->endOfWeek()],
            'This Month' => [now()->startOfMonth(), now()->endOfMonth()],
            'Last Month' => [now()->subMonthNoOverflow()->startOfMonth(), now()->subMonthNoOverflow()->endOfMonth()],
            'This Year (YTD)' => [now()->startOfYear(), now()->endOfDay()],
            'Last Year' => [now()->subYear()->startOfYear(), now()->subYear()->endOfYear()],
        ];

        $permitted = method_exists($user, 'permitted_locations') ? $user->permitted_locations($businessId) : 'all';
        $applyLoc = function($q) use ($permitted) {
            if ($permitted !== 'all' && is_array($permitted) && !empty($permitted)) { $q->whereIn('location_id', $permitted); }
        };

        $inserted = 0;
        foreach ($ranges as $label => [$start,$end]) {
            // Sales
            $qSell = \App\Transaction::where('business_id', $businessId)->where('type','sell')->where('status','final')->whereBetween('transaction_date', [$start, $end]);
            $applyLoc($qSell);
            $salesTotal = (float) $qSell->sum('final_total');
            AiMessage::create([
                'user_id' => $user->getAuthIdentifier(),
                'business_id' => $businessId,
                'supplier_id' => null,
                'scope' => 'business',
                'intent' => 'sales',
                'time_label' => $label,
                'normalized_query' => 'sales|business|'.$label,
                'question' => 'Precomputed '.$label.' sales total',
                'answer' => 'Php '.number_format($salesTotal, 2),
                'ok' => 1,
            ]); $inserted++;

            // Purchases
            $qPurch = \App\Transaction::where('business_id', $businessId)->whereIn('type',[ 'purchase','purchase_order'])
                ->whereBetween('transaction_date', [$start, $end]);
            $applyLoc($qPurch);
            $purchTotal = (float) $qPurch->sum('final_total');
            AiMessage::create([
                'user_id' => $user->getAuthIdentifier(),
                'business_id' => $businessId,
                'supplier_id' => null,
                'scope' => 'business',
                'intent' => 'purchases',
                'time_label' => $label,
                'normalized_query' => 'purchases|business|'.$label,
                'question' => 'Precomputed '.$label.' purchase total',
                'answer' => 'Php '.number_format($purchTotal, 2),
                'ok' => 1,
            ]); $inserted++;
        }

        return $request->wantsJson()
            ? response()->json(['ok' => true, 'inserted' => $inserted])
            : redirect()->route('ai.dashboard')->with('status', ['success' => 1, 'msg' => 'Reports learned: '.$inserted.' rows']);
    }
    public function index(Request $request)
    {
        $user = $request->user();
        if (!$user) {
            abort(401);
        }

        return view('ai.chat');
    }

    /**
     * Update AI chat language preference (session-based).
     */
    public function setLang(Request $request)
    {
        $request->validate([
            'lang' => ['required', 'in:en,tl,ceb'],
        ]);
        session(['ai_chat.lang_pref' => $request->string('lang')]);
        return response()->json(['ok' => true, 'lang' => $request->string('lang')]);
    }

    public function rate(Request $request)
    {
        $request->validate([
            'id' => ['required','integer'],
            'rating' => ['required','in:1,-1'],
        ]);
        $user = $request->user();
        $id = (int) $request->input('id');
        $rating = (int) $request->input('rating');
        try {
            $row = AiMessage::where('id', $id)->where('user_id', $user->getAuthIdentifier())->first();
            if (!$row) { return response()->json(['ok' => false], 404); }
            $row->rating = $rating;
            $row->save();
        } catch (\Throwable $e) {
            return response()->json(['ok' => false], 500);
        }
        return response()->json(['ok' => true]);
    }

    // Daily KJV Bible verse (cached per day)
    public function dailyVerse(Request $request)
    {
        $today = now()->format('Y-m-d');
        $cacheKey = 'daily_kjv_verse_' . $today;
        $val = Cache::get($cacheKey);
        if (!$val) {
            $val = $this->fetchDailyKjvVerse();
            // ensure structure
            if (!is_array($val) || empty($val['text']) || empty($val['reference'])) {
                $val = $this->fallbackKjvVerse();
            }
            Cache::put($cacheKey, $val, now()->endOfDay());
        }
        return response()->json(['ok' => true] + $val);
    }

    protected function fetchDailyKjvVerse(): array
    {
        // Try Bible API (random KJV verse)
        $endpoints = [
            'https://bible-api.com/?random=verse&translation=kjv',
            'https://bible-api.com/john%203:16?translation=kjv', // fallback specific
        ];
        foreach ($endpoints as $ep) {
            try {
                $ch = curl_init($ep);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_TIMEOUT, 10);
                $raw = curl_exec($ch);
                curl_close($ch);
                if (!is_string($raw) || $raw === '') continue;
                $data = json_decode($raw, true);
                if (!is_array($data)) continue;
                // bible-api returns either 'text' and 'reference' or an array of 'verses'
                $text = $data['text'] ?? null;
                $ref = $data['reference'] ?? null;
                if (!$text && !empty($data['verses']) && is_array($data['verses'])) {
                    $first = $data['verses'][0] ?? null;
                    if (is_array($first)) {
                        $text = $first['text'] ?? null;
                        if (isset($first['book_name'],$first['chapter'],$first['verse'])) {
                            $ref = $first['book_name'].' '.$first['chapter'].':'.$first['verse'];
                        }
                    }
                }
                if ($text && $ref) {
                    return ['text' => trim($text), 'reference' => trim($ref), 'translation' => 'KJV'];
                }
            } catch (\Throwable $e) { /* ignore */ }
        }
        return [];
    }

    protected function fallbackKjvVerse(): array
    {
        $verses = [
            ['ref' => 'John 3:16', 'text' => 'For God so loved the world, that he gave his only begotten Son, that whosoever believeth in him should not perish, but have everlasting life.'],
            ['ref' => 'Psalm 23:1', 'text' => 'The LORD is my shepherd; I shall not want.'],
            ['ref' => 'Proverbs 3:5-6', 'text' => 'Trust in the LORD with all thine heart; and lean not unto thine own understanding. In all thy ways acknowledge him, and he shall direct thy paths.'],
            ['ref' => 'Romans 8:28', 'text' => 'And we know that all things work together for good to them that love God, to them who are the called according to his purpose.'],
            ['ref' => 'Philippians 4:13', 'text' => 'I can do all things through Christ which strengtheneth me.'],
            ['ref' => 'Jeremiah 29:11', 'text' => 'For I know the thoughts that I think toward you, saith the LORD, thoughts of peace, and not of evil, to give you an expected end.'],
            ['ref' => 'Matthew 6:33', 'text' => 'But seek ye first the kingdom of God, and his righteousness; and all these things shall be added unto you.'],
        ];
        $idx = (int) (now()->format('z')) % count($verses);
        return ['text' => $verses[$idx]['text'], 'reference' => $verses[$idx]['ref'], 'translation' => 'KJV'];
    }

    public function message(Request $request, AiContextService $context, AiSchemaService $schema)
    {
        $request->validate([
            'message' => ['required', 'string', 'max:4000'],
        ]);

        $user = $request->user();
        if (!$user) {
            abort(401);
        }

        $userMessage = trim((string) $request->input('message'));

        // Language preference: prioritize Tagalog (tl), Cebuano/Bisaya (ceb), and English (en)
        $langPref = (string) session('ai_chat.lang_pref', '');
        $qLower = strtolower($userMessage);
        $askedCebu = (bool) preg_match('/\b(binisaya|bisaya|cebuano|cebu\s*language|mag\s*bisaya|speak\s*(bisaya|cebuano))\b/i', $userMessage);
        $askedTag = (bool) preg_match('/\b(tagalog|filipino|mag\s*tagalog|magsalita\s*ng\s*tagalog|speak\s*tagalog)\b/i', $userMessage);
        $askedEng = (bool) preg_match('/\b(english|ingles|speak\s*english)\b/i', $userMessage);
        if ($askedCebu) { $langPref = 'ceb'; }
        if ($askedTag) { $langPref = 'tl'; }
        if ($askedEng)  { $langPref = 'en'; }
        // If no explicit request and no stored pref, try a lightweight auto-detect among tl/ceb/en
        if ($langPref === '') {
            $looksTL = (bool) preg_match('/\b(pwede|po\b|opo\b|salamat|magkano|kailan|saan|meron|wala|gusto|para|paki)\b/i', $userMessage);
            $looksCEB = (bool) preg_match('/\b(unsa|pila|kaayo|salamat|kuan|dili|oo|wala|asa|pwede|ba|kani|kana|diria|gi|maka)\b/i', $userMessage);
            if ($looksCEB && !$looksTL) { $langPref = 'ceb'; }
            elseif ($looksTL && !$looksCEB) { $langPref = 'tl'; }
            else { $langPref = 'en'; }
        }
        session(['ai_chat.lang_pref' => $langPref]);

        // Currency preferences for responses
        $curr = (array) session('currency', []);
        $currencyCode = (string) ($curr['code'] ?? 'PHP');
        $currencySymbol = (string) ($curr['symbol'] ?? '₱');
        $symbolPlacement = (string) ($curr['symbol_placement'] ?? 'before');
        // Normalize PHP currency to Peso symbol if session symbol is missing or garbled
        if ($currencyCode === 'PHP') {
            $sym = trim($currencySymbol);
            if ($sym === '' || $sym === '$' || preg_match('/\x{FFFD}/u', $sym)) {
                $currencySymbol = '₱';
            }
        }

        $systemPrompt = "You are Dev Kim AI, a helpful assistant embedded in a business management system. Use a warm, friendly, and concise tone. For system questions (sales, purchases, stock, users, activity), you receive an authoritative Context block.\n- Treat any Context block as ground truth about the system.\n- When the Context provides numbers (totals, counts, quantities), answer directly using them.\n- Do NOT say you do not have access; if the Context is present, you DO have the necessary data.\n- If no Context is provided for a system question, say 'Not enough information' in one short sentence.\n- Keep answers short and actionable.\n- For general non-system questions, answer normally.\n\nCurrency rules: Use " . $currencyCode . " (symbol '" . $currencySymbol . "'). Place the symbol " . ($symbolPlacement === 'before' ? 'before' : 'after') . " the numbers without a trailing currency name. Example: '" . ($symbolPlacement === 'before' ? $currencySymbol . "1,234.56" : "1,234.56" . $currencySymbol) . "'. Never use $ or other currencies.";
        $systemPrompt .= "\n\nIdentity rules:\n- Your name is Dev Kim AI.\n- If a user asks who created you (e.g., 'who created/made/built you?'), answer exactly: The one and only Handsome Dev Kim Ian Lacson.";

        // Language rules block for the model
        $langHint = $langPref === 'tl' ? 'Tagalog (Filipino)'
            : ($langPref === 'ceb' ? 'Cebuano (Bisaya)' : 'English');
        $systemPrompt .= "\n\nLanguage rules:\n- Priority languages: Tagalog, Cebuano (Bisaya), English.\n- Current preference: " . $langHint . ".\n- Respond entirely in the preferred language unless the user explicitly asks to switch.\n- If the user asks to switch language (e.g., 'mag Tagalog', 'speak Bisaya', 'speak English'), switch and continue using that language until changed again.\n- Keep tone warm, friendly, and concise.";

        // Page hints: allow auto-scoping to a supplier shown on the current page
        $pageSupplierId = (int) ($request->input('page_supplier_id') ?? $request->header('X-Page-Supplier-Id') ?? session('ai.page_supplier_id') ?? 0);
        $pageSupplierName = (string) ($request->input('page_supplier_name') ?? $request->header('X-Page-Supplier-Name') ?? session('ai.page_supplier_name') ?? '');
        if ($pageSupplierId > 0) { session(['ai.page_supplier_id' => $pageSupplierId]); }
        if ($pageSupplierName !== '') { session(['ai.page_supplier_name' => $pageSupplierName]); }

        // Intent inference for elliptical follow-ups like "how about this month"
        $wantPurch = (bool) preg_match('/\b(purchase|purchases|buy|bought|procurement|po|purchase order)\b/i', $userMessage);
        $wantSales = (bool) preg_match('/\b(sale|sales|sell|revenue|invoice)\b/i', $userMessage);
        $lastIntent = (string) session('ai_chat.last_intent', '');
        if ($wantPurch) { session(['ai_chat.last_intent' => 'purchases']); }
        if ($wantSales) { session(['ai_chat.last_intent' => 'sales']); }

        $userMessageForContext = $userMessage;
        if (!$wantPurch && !$wantSales && ($lastIntent === 'purchases' || $lastIntent === 'sales')) {
            $userMessageForContext .= $lastIntent === 'purchases' ? ' purchases' : ' sales';
        }

        $contextText = $context->buildContextForUser($user, $userMessageForContext, [
            'page_supplier_id' => $pageSupplierId,
            'page_supplier_name' => $pageSupplierName,
        ]);

        // Optionally include schema summary for admins to help generic questions
        $schemaBlock = '';
        try {
            $roles = method_exists($user, 'getRoleNames') ? collect($user->getRoleNames())->map(function($r){ return strtolower(trim($r)); }) : collect();
            $isAdmin = $roles->contains('admin') || $roles->contains('super admin') || $roles->contains('superadmin') || $roles->contains('admin#' . strtolower((string)($user->business_id ?? '')));
        } catch (\Throwable $e) { $isAdmin = false; }
        if ($isAdmin) {
            $schemaBlock = $schema->getSchemaSummary(40, 10);
        }

        $history = (array) session('ai_chat.history', []);
        $trimmedHistory = array_slice($history, -10); // keep last 10 turns

        $messages = [];
        $messages[] = ['role' => 'system', 'content' => $systemPrompt];
        if ($contextText !== '') {
            $messages[] = ['role' => 'system', 'content' => "Context (user-scoped):\n" . $contextText];
        }
        if ($schemaBlock !== '') {
            $messages[] = ['role' => 'system', 'content' => $schemaBlock];
        }
        foreach ($trimmedHistory as $h) {
            if (!empty($h['role']) && !empty($h['content'])) {
                $messages[] = ['role' => $h['role'], 'content' => $h['content']];
            }
        }
        $messages[] = ['role' => 'user', 'content' => $userMessage];

        // Short-circuit identity and small-talk without calling OpenAI (friendly tone, localized)
        $q = strtolower($userMessage);
        if (preg_match('/\b(who\s+(created|made|built)\s+you)\b/', $q)) {
            $responseText = 'The one and only Handsome Dev Kim Ian Lacson';
        } elseif (preg_match('/\b(what\s+is\s+your\s+name|your\s+name)\b/', $q)) {
            $responseText = ($langPref === 'tl') ? 'Ako si Dev Kim AI.' : (($langPref === 'ceb') ? 'Ako si Dev Kim AI.' : 'Dev Kim AI');
        } elseif ($askedCebu) {
            $responseText = 'Oo, makasulti ko ug Cebuano. Sugdan nato sa Binisaya. 🙂';
        } elseif ($askedTag) {
            $responseText = 'Opo, makapagsasalita ako ng Tagalog. Magpapatuloy tayo sa Tagalog. 🙂';
        } elseif ($askedEng) {
            $responseText = 'Sure — I’ll continue in English. 🙂';
        } elseif (preg_match('/\b(hi|hello|hey|good\s+(morning|afternoon|evening)|kumusta|kamusta)\b/i', $userMessage)) {
            $responseText = ($langPref === 'tl') ? 'Hello! Paano kita matutulungan ngayon?' : (($langPref === 'ceb') ? 'Hello! Unsa akong ikatabang nimo karon?' : 'Hello! How can I help you today?');
        } elseif (preg_match('/\b(how\s+are\s+you)\b/i', $userMessage)) {
            $responseText = ($langPref === 'tl') ? 'Ayos naman ako at handang tumulong. Ano ang gusto mong i‑check?' : (($langPref === 'ceb') ? 'Maayo ra ko ug andam motabang. Unsa imong gusto susihon?' : "I'm doing great and ready to help. What would you like to check?");
        } elseif (preg_match('/\b(thank\s*you|thanks|salamat)\b/i', $userMessage)) {
            $responseText = ($langPref === 'tl') ? 'Walang anuman! Kung may iba ka pang kailangan, sabihin mo lang.' : (($langPref === 'ceb') ? 'Walay sapayan! Kung naa kay lain pang pangutana, ingna lang ko.' : 'You’re welcome! If you need anything else, just ask.');
        } elseif (preg_match('/\b(bye|goodbye|salamat\s+po)\b/i', $userMessage)) {
            $responseText = ($langPref === 'tl') ? 'Paalam! Magandang araw!' : (($langPref === 'ceb') ? 'Bye! Maayong adlaw!' : 'Goodbye! Have a wonderful day!');
        } else {
            // 1) Deterministic price answer first so FAQs don't override product queries
            if (!isset($responseText)) {
                try {
                    $priceAns = $this->deterministicPriceAnswer($user, $userMessage);
                    if (is_string($priceAns) && $priceAns !== '') {
                        $responseText = $priceAns;
                    }
                } catch (\Throwable $e) { /* ignore */ }
            }
            // 2) FAQs for general questions (skip for price-like questions)
            $looksPrice = (bool) preg_match('/\b(price|presyo|magkano|how\s*much|selling\s*price|group\s*price|srp|sku)\b/i', $userMessage);
            if (!isset($responseText) && !$looksPrice) {
                try {
                    $faq = app(\App\Services\FaqService::class);
                    $faqAnswer = $faq->answerFor($userMessage);
                    if (is_string($faqAnswer) && $faqAnswer !== '') { $responseText = $faqAnswer; }
                } catch (\Throwable $e) { /* ignore */ }
            }
            // If the parsed context already has a numeric answer and the question is a totals-style query
            $responseText = $responseText ?? null;
            $looksNumericTotals = (bool) preg_match('/\b(total|amount|how\s+much|sum|grand\s+total)\b/i', $userMessageForContext);
            $parsedEarly = $this->parseFromContext($userMessageForContext, $contextText, $lastIntent);
            if ($responseText === null && $looksNumericTotals && $parsedEarly) {
                $responseText = $this->fallbackFromParsed($parsedEarly);
            }
            if ($responseText === null) {
                $responseText = $this->callOpenAI($messages);
            }
            // Fallback: if model failed/empty, try to answer deterministically from Context
            $parsed = $parsedEarly ?: $this->parseFromContext($userMessageForContext, $contextText, $lastIntent);
            if (!is_string($responseText) || $responseText === '' || stripos($responseText, 'could not generate') !== false || stripos($responseText, 'temporary error') !== false) {
                $fallback = $this->fallbackFromParsed($parsed);
                if ($fallback === null) {
                    // Try cache reuse if we can derive a normalized key
                    $intentF = $parsed['type'] ?? ($wantPurch ? 'purchases' : ($wantSales ? 'sales' : null));
                    $labelF = $parsed['label'] ?? $this->deriveTimeLabel($q);
                    $scopeF = $parsed['scope_key'] ?? (session('ai_chat.last_scope') ?: null);
                    $supplierIdF = (int) ($pageSupplierId ?: (session('ai.page_supplier_id') ?? 0));
                    $cached = $this->getCachedAnswer((int) ($user->business_id ?? 0), $supplierIdF ?: null, $scopeF, $intentF, $labelF);
                    if ($cached !== null) { $fallback = 'Php ' . $cached; }
                }
                if ($fallback !== null) {
                    $responseText = $fallback;
                }
            }
        }

        // Update session history
        $history[] = ['role' => 'user', 'content' => $userMessage];
        $history[] = ['role' => 'assistant', 'content' => $responseText];
        session(['ai_chat.history' => array_slice($history, -20)]);

        // Persist message for learning/cache analytics
        $savedMessageId = null;
        try {
            $intent = $parsed['type'] ?? ($wantPurch ? 'purchases' : ($wantSales ? 'sales' : 'general'));
            $timeLabel = $parsed['label'] ?? null;
            $scope = $parsed['scope_key'] ?? null; // supplier | business | your | user
            $supplierId = (int) ($pageSupplierId ?: (session('ai.page_supplier_id') ?? 0));
            $row = AiMessage::create([
                'user_id' => $user->getAuthIdentifier(),
                'business_id' => (int) ($user->business_id ?? 0),
                'supplier_id' => $supplierId ?: null,
                'scope' => $scope,
                'intent' => $intent,
                'time_label' => $timeLabel,
                'normalized_query' => trim(($intent ?: '') . '|' . ($scope ?: '') . '|' . ($timeLabel ?: '')),
                'question' => $userMessage,
                'answer' => $responseText,
                'ok' => is_string($responseText) && $responseText !== '' ? 1 : 0,
            ]);
            $savedMessageId = $row->id ?? null;
        } catch (\Throwable $e) {
            // don't block on analytics failures
        }

        return response()->json([
            'ok' => true,
            'reply' => $responseText,
            'message_id' => $savedMessageId,
        ]);
    }

    /**
     * Direct DB-based price answer for queries like:
     * - "what is the regular price of Paracetamol"
     * - "price of <sku> (End-user)"
     * Returns a short formatted string or null if not detected.
     */
    protected function deterministicPriceAnswer($user, string $text): ?string
    {
        // normalize smart quotes and trim
        $text = strtr($text, [
            '“' => '"', '”' => '"', '‘' => "'", '’' => "'",
        ]);
        $q = strtolower($text);
        if (!preg_match('/price|how\s+much|presyo|magkano|selling\s+price|group\s*price|srp/i', $text)) {
            return null;
        }

        $businessId = (int) ($user->business_id ?? session('business.id'));
        if (!$businessId) return null;
        $permitted = method_exists($user, 'permitted_locations') ? $user->permitted_locations($businessId) : 'all';

        // Extract candidate product/sku
        $prodCandidate = null; $bySku = false;
        if (preg_match('/sku\s*[:#]?\s*([A-Za-z0-9._\-]{2,100})/i', $text, $m)) { $prodCandidate = $m[1]; $bySku = true; }
        // "price for <term>" or "price of <term>" or "magkano <term>"
        if ($prodCandidate === null && preg_match('/(?:price|presyo|magkano)\s+(?:for|ng|of)?\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})/i', $text, $m)) { $prodCandidate = trim($m[1]); }
        // "how much is (the) price for/of <term>" or "how much for <term>"
        if ($prodCandidate === null && preg_match('/how\s*much\s*(?:is\s*(?:the\s*)?)?(?:price\s*)?(?:for|of|ng)?\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})/i', $text, $m)) { $prodCandidate = trim($m[1]); }
        if ($prodCandidate === null && preg_match('/(?:for|para\s+sa)\s*([A-Za-z0-9 &_\-\(\)\'\"\/\.,]{2,120})\s*(?:price|presyo)?/i', $text, $m)) { $prodCandidate = trim($m[1]); }
        if ($prodCandidate === null) return null;

        // Resolve group by name/alias if mentioned
        $groups = \DB::table('selling_price_groups')->where('business_id', $businessId)->where('is_active',1)->pluck('name','id');
        $groupId = null; $groupLabel = null;
        foreach ($groups as $gid => $gname) { if (str_contains($q, strtolower($gname))) { $groupId=(int)$gid; $groupLabel=(string)$gname; break; } }
        if ($groupId === null) {
            $aliases = ['regular'=>'regular','end user'=>'end-user','end-user'=>'end-user','credit card'=>'credit card','credit'=>'credit','distributor'=>'distributor'];
            foreach ($aliases as $needle=>$canonical) {
                if (str_contains($q, $needle)) {
                    foreach ($groups as $gid=>$gname) { if (str_contains(strtolower($gname), $canonical)) { $groupId=(int)$gid; $groupLabel=(string)$gname; break 2; } }
                }
            }
        }

        $term = trim($prodCandidate);
        // If the user included a price group name alongside the product (e.g., "End-user"),
        // strip it from the product search term to avoid failed matches.
        $stripWords = ['regular','end user','end-user','credit','credit card','distributor','srp','selling price','price','presyo','magkano'];
        if (!empty($groupLabel)) { $stripWords[] = strtolower($groupLabel); }
        $tLower = strtolower($term);
        foreach ($stripWords as $sw) {
            $tLower = str_ireplace($sw, '', $tLower);
        }
        $tLower = preg_replace('/\s+/', ' ', $tLower);
        $tLower = trim($tLower, " \t\n\r\0\x0B-_/.,()[]{}?");
        if ($tLower !== '') { $term = $tLower; }
        $baseQ = \DB::table('products as p')
            ->join('variations as v','v.product_id','=','p.id')
            ->leftJoin('product_variations as pv','pv.id','=','v.product_variation_id')
            ->where('p.business_id', $businessId);
        if ($bySku) { $baseQ->where('v.sub_sku','like','%'.$term.'%'); }
        else { $baseQ->where(function($w) use ($term){ $w->where('p.name','like','%'.$term.'%')->orWhere('v.sub_sku','like','%'.$term.'%'); }); }
        $vars = $baseQ->limit(3)->get(['p.name as product_name','v.id as variation_id','v.name as variation_name','v.sub_sku','v.sell_price_inc_tax']);
        if ($vars->count() === 0) return null;

        $varIds = $vars->pluck('variation_id')->all();
        $vgp = \DB::table('variation_group_prices as vgp')
            ->join('selling_price_groups as spg','spg.id','=','vgp.price_group_id')
            ->whereIn('vgp.variation_id', $varIds)
            ->select('vgp.variation_id','spg.name as group_name','vgp.price_inc_tax','vgp.price_type','vgp.price_group_id')
            ->get();
        // Quantity on hand per variation (respect permitted locations)
        $qtyQ = \DB::table('variation_location_details as vld')
            ->whereIn('vld.variation_id', $varIds)
            ->select('vld.variation_id', \DB::raw('COALESCE(SUM(vld.qty_available),0) as qty'));
        if ($permitted !== 'all' && is_array($permitted) && !empty($permitted)) {
            $qtyQ->whereIn('vld.location_id', $permitted);
        }
        $qtyRows = $qtyQ->groupBy('vld.variation_id')->get();
        $byVar = [];
        foreach ($vgp as $r) { $byVar[(int)$r->variation_id][] = $r; }
        $qtyByVar = [];
        foreach ($qtyRows as $qr) { $qtyByVar[(int)$qr->variation_id] = (float) $qr->qty; }
        $util = new \App\Utils\Util();
        $tables = [];
        foreach ($vars as $vrow) {
            $base = (float) $vrow->sell_price_inc_tax;
            $varName = is_string($vrow->variation_name) ? trim($vrow->variation_name) : '';
            if ($varName !== '' && preg_match('/^(dummy|default|n\/a|na|none|-|single)$/i', $varName)) { $varName = ''; }
            $label = trim($vrow->product_name . ($varName ? (' - '.$varName) : '') . ' ['.($vrow->sub_sku ?: '-') . ']');
            $entries = $byVar[(int)$vrow->variation_id] ?? [];
            $rows = [];
            $qtyForVar = number_format($qtyByVar[(int)$vrow->variation_id] ?? 0, 0);
            // Fetch available lots with expiry and remaining qty (respect locations)
            $lotRowsQ = \DB::table('purchase_lines as pl')
                ->join('transactions as t','t.id','=','pl.transaction_id')
                ->where('t.business_id', $businessId)
                ->where('pl.variation_id', (int)$vrow->variation_id)
                ->select(
                    'pl.lot_number',
                    'pl.exp_date',
                    \DB::raw('(COALESCE(pl.quantity,0) - COALESCE(pl.quantity_sold,0) - COALESCE(pl.quantity_adjusted,0) - COALESCE(pl.quantity_returned,0) - COALESCE(pl.mfg_quantity_used,0)) as qty_remaining')
                )
                ->orderBy('pl.exp_date','asc');
            if ($permitted !== 'all' && is_array($permitted) && !empty($permitted)) {
                $lotRowsQ->whereIn('t.location_id', $permitted);
            }
            $lotRows = $lotRowsQ->leftJoin('business_locations as bl','bl.id','=','t.location_id')->get();
            $lotParts = [];
            $lotTableRows = [];
            foreach ($lotRows as $lr) {
                $rem = (float) ($lr->qty_remaining ?? 0);
                if ($rem <= 0.00001) continue;
                $ln = trim((string) ($lr->lot_number ?? ''));
                $ed = '';
                if (!empty($lr->exp_date)) { try { $ed = \Carbon\Carbon::parse($lr->exp_date)->toDateString(); } catch (\Throwable $e) { $ed = (string)$lr->exp_date; } }
                $piece = ($ln !== '' ? $ln : 'Lot') . ($ed !== '' ? (' exp ' . $ed) : '') . '=' . number_format($rem, 0);
                $lotParts[] = $piece;
                $locNm = trim((string) ($lr->name ?? 'Location'));
                $locCode = isset($lr->location_id) ? trim((string)$lr->location_id) : '';
                $locLabel = $locNm . ($locCode !== '' ? (' (' . $locCode . ')') : '');
                $lotTableRows[] = [ $locLabel, number_format($rem, 4), ($ln !== '' ? $ln : '—'), ($ed !== '' ? $ed : '—') ];
                if (count($lotParts) >= 6) break; // keep it compact
            }
            $lotsStr = $lotParts ? (' • Lots: ' . implode(', ', $lotParts)) : '';
            $caption = $label . ' • Quantity Available: ' . $qtyForVar . $lotsStr;
            // Always include default/base selling price as the first row
            $rows[] = ['Default selling price', 'Php ' . number_format($base,2)];
            if (!empty($entries) && $groupId !== null) {
                $match = collect($entries)->first(fn($x)=> (int)$x->price_group_id === (int)$groupId);
                if ($match) {
                    $price = ($match->price_type === 'percentage') ? $util->calc_percentage($base, (float)$match->price_inc_tax) : (float)$match->price_inc_tax;
                    $rows[] = [$groupLabel, 'Php ' . number_format($price,2)];
                } else {
                    foreach ($entries as $e) {
                        $price = ($e->price_type === 'percentage') ? $util->calc_percentage($base, (float)$e->price_inc_tax) : (float)$e->price_inc_tax;
                        $rows[] = [$e->group_name, 'Php ' . number_format($price,2)];
                    }
                }
            } elseif (!empty($entries)) {
                foreach ($entries as $e) {
                    $price = ($e->price_type === 'percentage') ? $util->calc_percentage($base, (float)$e->price_inc_tax) : (float)$e->price_inc_tax;
                    $rows[] = [$e->group_name, 'Php ' . number_format($price,2)];
                }
            }
            if (!empty($rows)) { $tables[] = ['caption' => $caption, 'rows' => $rows, 'lot_rows' => $lotTableRows]; }
        }
        if (empty($tables)) return null;
        $out = '';
        foreach ($tables as $t) {
            $out .= "::table::\n" . $t['caption'] . "\n" . 'Group|Price' . "\n";
            foreach ($t['rows'] as $r) { $out .= ($r[0] ?? '') . '|' . ($r[1] ?? '') . "\n"; }
            $out .= "::endtable::\n";
            if (!empty($t['lot_rows'])) {
                $out .= "::table::\n" . 'Available Lots' . "\n" . 'Location|Qty|Lot NO.|Exp.Date' . "\n";
                foreach ($t['lot_rows'] as $lr) { $out .= ($lr[0] ?? '') . '|' . ($lr[1] ?? '') . '|' . ($lr[2] ?? '') . '|' . ($lr[3] ?? '') . "\n"; }
                $out .= "::endtable::\n";
            }
        }
        // Include qty values in output lines
        $outLines = [];
        foreach ($tables as $t) {
            $outLines[] = '::table::';
            $outLines[] = $t['caption'];
            $outLines[] = 'Group|Price';
            foreach ($t['rows'] as $r) { $outLines[] = ($r[0] ?? '') . '|' . ($r[1] ?? ''); }
            $outLines[] = '::endtable::';
            if (!empty($t['lot_rows'])) {
                $outLines[] = '::table::';
                $outLines[] = 'Available Lots';
                $outLines[] = 'Location|Qty|Lot NO.|Exp.Date';
                foreach ($t['lot_rows'] as $lr) { $outLines[] = ($lr[0] ?? '') . '|' . ($lr[1] ?? '') . '|' . ($lr[2] ?? '') . '|' . ($lr[3] ?? ''); }
                $outLines[] = '::endtable::';
            }
        }
        return implode("\n", $outLines);
    }

    protected function callOpenAI(array $messages): string
    {
        $apiKey = (string) config('services.openai.key');
        $model = (string) config('services.openai.model', 'gpt-4o-mini');
        $endpoint = (string) config('services.openai.endpoint', 'https://api.openai.com/v1/chat/completions');

        if ($apiKey === '') {
            return 'OpenAI API key is not configured. Please set OPENAI_API_KEY in your .env.';
        }

        $payload = [
            'model' => $model,
            'messages' => $messages,
            'temperature' => 0.2,
            'max_tokens' => 800,
        ];

        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey,
        ]);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $raw = curl_exec($ch);
        $errno = curl_errno($ch);
        $err = curl_error($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($errno) {
            Log::warning('OpenAI cURL error', ['errno' => $errno, 'error' => $err]);
            return 'There was a temporary error contacting the AI. Please try again.';
        }

        $data = json_decode((string) $raw, true);
        if (!is_array($data)) {
            Log::warning('OpenAI invalid response', ['status' => $status, 'raw' => Str::limit((string)$raw, 500)]);
            return 'The AI returned an unexpected response.';
        }

        $text = $data['choices'][0]['message']['content'] ?? null;
        if (!is_string($text) || $text === '') {
            Log::warning('OpenAI empty content', ['status' => $status, 'data' => $data]);
            return 'I could not generate a response from the provided context.';
        }

        return $text;
    }

    /**
     * Deterministic fallback: parse Context block for totals and answer directly.
     * Supports purchases/sales summaries emitted by AiContextService.
     */
    protected function parseFromContext(string $userMessage, string $contextText, string $lastIntent = ''): ?array
    {
        $q = strtolower($userMessage);
        if ($contextText === '') { return null; }

        // Decide whether the user asked about purchases or sales
        $wantPurch = (bool) preg_match('/\b(purchase|purchases|buy|bought|procurement|po|purchase order)\b/i', $userMessage);
        $wantSales = (bool) preg_match('/\b(sale|sales|sell|revenue|invoice)\b/i', $userMessage);
        if (!$wantPurch && !$wantSales) {
            if ($lastIntent === 'purchases') { $wantPurch = true; }
            elseif ($lastIntent === 'sales') { $wantSales = true; }
        }

        // Time hints
        $timeHints = [];
        if (preg_match('/today/i', $q)) { $timeHints[] = 'Today'; }
        if (preg_match('/this\s+week|week/i', $q)) { $timeHints[] = 'This Week'; }
        if (preg_match('/last\s+month|previous\s+month/i', $q)) { $timeHints[] = 'Last Month'; }
        if (preg_match('/this\s+month|\bmonth\b/i', $q)) { $timeHints[] = 'This Month'; }
        if (preg_match('/last\s+year|previous\s+year/i', $q)) { $timeHints[] = 'Last Year'; }
        if (preg_match('/this\s+year|year\s+to\s+date|ytd|current\s+year/i', $q)) { $timeHints[] = 'This Year'; $timeHints[] = 'This Year (YTD)'; }

        $lines = preg_split('/\r?\n/', $contextText);
        if (!is_array($lines)) { return null; }

        // Build regex for either purchases or sales
        $patternPurch = '/^(Today|This Week|Last Month|This Month|Last Year|This Year(?: \(YTD\))?)\s+Purchases\s*\(([^)]*)\):\s*total=Php\s*([0-9,.]+),\s*count=([0-9]+)/i';
        $patternSales = '/^(Today|This Week|Last Month|This Month|Last Year|This Year(?: \(YTD\))?)\s+Sales\s*\(([^)]*)\):\s*total=Php\s*([0-9,.]+),\s*count=([0-9]+)/i';
        $patternProducts = '/^(Product Stock|Inventory Overview.*):\s*(.+)$/i';
        $patternLowStock = '/^(Low Stock):\s*(.+)$/i';

        $candidates = [];
        foreach ($lines as $line) {
            $m = [];
            if ($wantPurch && preg_match($patternPurch, $line, $m)) {
                $candidates[] = ['type' => 'purchases', 'label' => $m[1], 'scope' => $m[2], 'total' => $m[3], 'count' => $m[4]];
            }
            if ($wantSales && preg_match($patternSales, $line, $m)) {
                $candidates[] = ['type' => 'sales', 'label' => $m[1], 'scope' => $m[2], 'total' => $m[3], 'count' => $m[4]];
            }
            if (preg_match($patternProducts, $line, $m)) {
                // Always accept product summaries as candidates for fallback
                $candidates[] = ['type' => 'products', 'label' => $m[1], 'scope' => null, 'text' => trim($m[1] . ': ' . $m[2])];
            }
            if (preg_match($patternLowStock, $line, $m)) {
                $candidates[] = ['type' => 'products', 'label' => $m[1], 'scope' => null, 'text' => trim($m[1] . ': ' . $m[2])];
            }
        }
        if (empty($candidates)) { return null; }

        // Prefer candidates that match the time hints; else return the first
        foreach ($timeHints as $hint) {
            foreach ($candidates as $c) {
                if (strcasecmp($c['label'], $hint) === 0) {
                    $c['scope_key'] = $this->scopeKeyFromLabel($c['scope']);
                    return $c;
                }
            }
        }
        $top = $candidates[0];
        if (isset($top['scope'])) { $top['scope_key'] = $this->scopeKeyFromLabel((string)$top['scope']); }
        return $top;
    }

    protected function fallbackFromParsed(?array $parsed): ?string
    {
        if (!$parsed) return null;
        if (($parsed['type'] ?? '') === 'products') {
            return (string) ($parsed['text'] ?? '');
        }
        if (isset($parsed['total'])) {
            return 'Php ' . ($parsed['total'] ?? '0.00');
        }
        return null;
    }

    protected function scopeKeyFromLabel(string $scopeLabel): ?string
    {
        $s = strtolower($scopeLabel);
        if (str_starts_with($s, 'supplier')) return 'supplier';
        if ($s === 'business') return 'business';
        if ($s === 'your') return 'your';
        if (str_starts_with($s, 'user')) return 'user';
        return null;
    }

    protected function deriveTimeLabel(string $q): ?string
    {
        if (preg_match('/today/i', $q)) return 'Today';
        if (preg_match('/this\s+week|week/i', $q)) return 'This Week';
        if (preg_match('/last\s+month|previous\s+month/i', $q)) return 'Last Month';
        if (preg_match('/this\s+month|\bmonth\b/i', $q)) return 'This Month';
        if (preg_match('/last\s+year|previous\s+year/i', $q)) return 'Last Year';
        if (preg_match('/this\s+year|year\s+to\s+date|ytd|current\s+year/i', $q)) return 'This Year (YTD)';
        return null;
    }

    protected function getCachedAnswer(int $businessId, ?int $supplierId, ?string $scope, ?string $intent, ?string $timeLabel): ?string
    {
        try {
            if (!$businessId || !$intent || !$timeLabel) return null;
            $norm = trim(($intent ?: '') . '|' . ($scope ?: '') . '|' . ($timeLabel ?: ''));
            $row = AiMessage::query()
                ->where('business_id', $businessId)
                ->when($supplierId, function ($q) use ($supplierId) {
                    return $q->where('supplier_id', $supplierId);
                })
                ->where('normalized_query', $norm)
                ->where('ok', 1)
                ->latest('id')
                ->first(['answer','created_at']);
            if (!$row) return null;
            $ttl = 600; // default 10 min
            if ($timeLabel === 'Today') $ttl = 300;
            elseif ($timeLabel === 'This Week') $ttl = 900;
            elseif ($timeLabel === 'This Month') $ttl = 900;
            elseif ($timeLabel === 'This Year (YTD)') $ttl = 1800;
            elseif ($timeLabel === 'Last Month' || $timeLabel === 'Last Year') $ttl = 86400; // mostly stable
            if (now()->diffInSeconds($row->created_at) > $ttl) return null;
            // Extract numeric
            if (preg_match('/Php\s*([0-9,.]+)/i', (string)$row->answer, $m)) {
                return (string)$m[1];
            }
            return null;
        } catch (\Throwable $e) { return null; }
    }
}

